#! /usr/bin/env python3
# -*- coding: utf-8 -*-

#---------------------------------------------------------------------------------------------------
# File: f2579.py
#
# Sub: Designing rectangle array with random spacing and height
# Ref.: https://www.klayout.de/forum/discussion/2579/
# Posted by: Krishp
#
# Author: Kazzz-S
# Last modified: 2024-09-03
#---------------------------------------------------------------------------------------------------
import os
import sys
import pandas as pd
import numpy  as np
import platform
import optparse
import copy
from   klayout import db, tl

#------------------------------------------------------------------------------
## To split the file name extensions up to the second dot from the tail
#
# @param[in] fname  file name
#
# @return (path, ext)-tuple
#------------------------------------------------------------------------------
def SplitExtensions(fname):
  path1, ext1 = os.path.splitext( os.path.basename(fname) )
  path2, ext2 = os.path.splitext( os.path.basename(path1) )

  if ext2 == "":
    return (path1, ext1)
  else:
    return (path2, ext2+ext1)

#------------------------------------------------------------------------------
## To test if file names have a valid extension
#
# @param[in] files      file name list
# @param[in] validext   valid (expected) file name extension like '.gds.gz'
#
# @return True if all the files have the valid extension; False, otherwise
#------------------------------------------------------------------------------
def FileHasValidExtension( files, validext ):
  files = copy.deepcopy(files)

  if isinstance( validext, str ):
    validext = [ validext ]
  elif not isinstance( validext, list ):
    raise Exception( "! validext must be a single string or list of strings" )
  validext = [ item.upper() for item in validext ]

  for candidate in validext:
    for item in files:
      path, ext = SplitExtensions(item)
      if ext.upper() == candidate:
        files.remove(item)
        continue

  if len(files) == 0:
    return True
  else:
    return False

#------------------------------------------------------------------------------
## To set global variables
#------------------------------------------------------------------------------
def SetGlobals():
  global DebugLevel     # debug level
  global Usage          # string on usage
  global ExcelFile      # input Excel file
  global BottomLeft     # the bottom-left coordinate in [um]
  global TopCellName    # the top cell name
  global LayerDtype     # the layer, dtype
  global OutputFile     # the output design file name
  # auxiliary variables on platform
  global System         # 6-tuple from platform.uname()
  global Node           # - do -
  global Release        # - do -
  global Version        # - do -
  global Machine        # - do -
  global Processor      # - do -

  Usage  = "\n"
  Usage += "-------------------------------------------------------------------------------------\n"
  Usage += "f2579.py\n"
  Usage += "  << To design rectangle array with random spacing and height >>\n"
  Usage += "     See https://www.klayout.de/forum/discussion/2579/ for requirements.\n"
  Usage += "\n"
  Usage += "$ [python3] f2579.py\n"
  Usage += "   option & argument : comment on option if any           default value\n"
  Usage += "   ------------------------------------------------------+-----------------------\n"
  Usage += "    <-e|--excel <*.xlsx>>: Excel file with one sheet     | ''\n"
  Usage += "    [-b|--BL <x,y>]: bottom-left coordinate in [um]      | '0,0'\n"
  Usage += "    [-t|--topcell <name>]: top cell name                 | TOP\n"
  Usage += "    [-l|--layer <L,D>]: layer, dtype                     | '1,0'\n"
  Usage += "    [-o|--output <design_file>]: output file name        | derived from the Excel\n"
  Usage += "    [-d|--debug  <level>]                                | 0\n"
  Usage += "    [-?|--?] : print this usage and exit                 | disabled\n"
  Usage += "---------------------------------------------------------+-----------------------\n"
  Usage += "  [example] $ ./f2579.py -e Scenario-1.xlsx -d 1\n"
  Usage += "-------------------------------------------------------------------------------------\n"

  DebugLevel  = 0
  ExcelFile   = ""
  BottomLeft  = (0.0, 0.0)
  TopCellName = "TOP"
  LayerDtype  = (1, 0)
  OutputFile  = ""

  # This script has no system-dependent operations; it's just a habit.
  (System, Node, Release, Version, Machine, Processor) = platform.uname()

#------------------------------------------------------------------------------
## To parse the command line arguments
#------------------------------------------------------------------------------
def ParseCommandLineArguments():
  global DebugLevel
  global Usage
  global ExcelFile
  global BottomLeft
  global TopCellName
  global LayerDtype
  global OutputFile

  p = optparse.OptionParser( usage=Usage )

  p.add_option( '-e', '--excel',
                dest='excel',
                help='Excel file with one sheet' )

  p.add_option( '-b', '--BL',
                dest='bottom_left',
                help='bottom-left coordinate in [um]' )

  p.add_option( '-t', '--topcell',
                dest='topcell',
                help='top cell name' )

  p.add_option( '-l', '--layer',
                dest='layer',
                help='layer, dtype' )

  p.add_option( '-o', '--output',
                dest='output',
                help='output file name' )

  p.add_option( '-d', '--debug',
                dest='debug_level',
                help='debug level (1)' )

  p.add_option( '-?', '--??',
                action='store_true',
                dest='checkusage',
                default=False,
                help='check usage' )

  p.set_defaults( excel       = "",
                  bottom_left = "0.0,0.0",
                  topcell     = "TOP",
                  layer       = "1,0",
                  output      = "",
                  debug_level = "0",
                  checkusage  = False )

  opt, args = p.parse_args()
  if opt.checkusage:
    print(Usage)
    sys.exit(0)

  DebugLevel = int(opt.debug_level)
  ExcelFile  = opt.excel
  if ExcelFile == "":
    print( "! Input Excel file is mandatory" )
    print(Usage)
    sys.exit(1)
  if not os.path.isfile(ExcelFile):
    print( f"! Excel file <{ExcelFile}> not found" )
    print(Usage)
    sys.exit(1)
  if not FileHasValidExtension( [ExcelFile], [".xlsx"] ):
    print( f"! Excel file <{ExcelFile}> must have the extension .xlsx" )
    print(Usage)
    sys.exit(1)

  try:
    x, y = opt.bottom_left.split(",")
  except ValueError:
    print( f"! Invalid bottom_left coordinate {opt.bottom_left}" )
    print(Usage)
    sys.exit(1)
  else:
    BottomLeft = (float(x), float(y))

  TopCellName = opt.topcell
  if TopCellName == "":
    print( "! Top cell name is mandatory" )
    print(Usage)
    sys.exit(1)

  try:
    l, d = opt.layer.split(",")
  except ValueError:
    print( f"! Invalid layer, dtype {opt.layer}" )
    print(Usage)
    sys.exit(1)
  else:
    LayerDtype = (int(l), int(d))

  OutputFile = opt.output
  if OutputFile == "":
    OutputFile = ExcelFile.replace( ".xlsx", ".gds" )
  if not FileHasValidExtension( [OutputFile], [".gds", ".oas"] ):
    print( f"! Output file <{OutputFile}> must have the extension [.gds, .oas]" )
    print(Usage)
    sys.exit(1)

#------------------------------------------------------------------------------
## To parse an Excel file that controls the design and then generate the design
#------------------------------------------------------------------------------
def ProcessExellFile():
  #--------------------------------------------------------------
  # [1] Read the Excel file and process the empty "space" (NaN)
  #--------------------------------------------------------------
  df = pd.read_excel( ExcelFile, comment="#" )
  if DebugLevel > 0:
    print(ExcelFile)
    print(df)
    print("")

  dicFloor = dict()
  for idx in range(len(df)):
    floor  = int  (df.iloc[idx,0])
    width  = float(np.round(df.iloc[idx,1], 1))
    height = float(np.round(df.iloc[idx,2], 1))
    space  = float(np.round(df.iloc[idx,3], 1))
    if np.isnan(space):
      dicFloor[idx] = [floor, width, height, 0.0]
    else:
      dicFloor[idx] = [floor, width, height, space]
    if DebugLevel > 1:
      print(dicFloor[idx])

  #--------------------------------------------------------------
  # [2] Construct rectangles
  #--------------------------------------------------------------
  x0, y0  = BottomLeft
  dicRect = dict()
  for idx in sorted(dicFloor.keys()):
    if idx == 0:
      lowerHeightSum = 0.0
    else:
      lowerHeightSum = 0.0
      for x in range(idx):
        [floor, width, height, space] = dicFloor[x]
        lowerHeightSum += height + space
    [floor, width, height, space] = dicFloor[idx]
    dicRect[idx] = db.DBox(x0,       y0+space+lowerHeightSum,
                           x0+width, y0+space+lowerHeightSum+height)

  #--------------------------------------------------------------
  # [3] Prepare Layout and populate it with the rectangles
  #--------------------------------------------------------------
  layout  = db.Layout()
  l, d    = LayerDtype
  topcell = layout.create_cell(TopCellName)
  for idx in sorted(dicRect.keys()):
    topcell.shapes(layout.layer(l, d)).insert(dicRect[idx])

  #--------------------------------------------------------------
  # [4] Write the output file
  #--------------------------------------------------------------
  layout.write(OutputFile)

#------------------------------------------------------------------------------
## The main function
#------------------------------------------------------------------------------
def main():
  #-----------------------------------------------------------------------
  # [1] Set the global objects
  #-----------------------------------------------------------------------
  rt = tl.Timer()
  rt.start()
  SetGlobals()

  #-----------------------------------------------------------------------
  # [2] Parse the command line arguments
  #-----------------------------------------------------------------------
  ParseCommandLineArguments()

  #-----------------------------------------------------------------------
  # [3] Run!
  #-----------------------------------------------------------------------
  ProcessExellFile()

  #-----------------------------------------------------------------------
  # [4] Print the elapsed time
  #-----------------------------------------------------------------------
  rt.stop()
  lead  = ">>> "
  etime = "{0:s}Elapsed Time = {1:.3f} [sec] {2:s}".format(lead, rt.wall(), lead)
  print(etime)

#===================================================================================
if __name__ == "__main__":
  main()

#---------------
# End of file
#---------------
