#!/usr/bin/env python3
# -*- coding: utf-8 -*-

from __future__ import annotations

import os
import sys
import json
import time
import tempfile
import shutil
import subprocess
import traceback
from pathlib import Path
from typing import Optional, Any, Dict, Tuple
import xml.etree.ElementTree as ET

from urllib.parse import urljoin
from urllib.request import Request, urlopen
from urllib.error import HTTPError, URLError

from PySide6 import QtCore, QtWidgets


APP_NAME = "GDS Launcher"
ORG_NAME = "Forum2834"
APP_SETTINGS_NAME = "GDSLauncher"

DEFAULT_KLAYOUT_CMD = "klayout"

BASE_DIR = Path(__file__).resolve().parent
TESTDATA_DIR = BASE_DIR / "testdata"

LOCAL_MAP = {
    "TEST-GDS-001": {
        "gds": TESTDATA_DIR / "sample.gds",
        "lyp": TESTDATA_DIR / "sample.lyp",
    },
    "TEST-GDS-002": {
        "gds": TESTDATA_DIR / "cobra3b.gds",
        "lyp": TESTDATA_DIR / "cobra3b.lyp",
    },
}

API_LOGIN = "/api/auth/login"
API_LOOKUP = "/api/layouts/by-ref/{gdsRefId}"

TEMP_PREFIX = "gdslauncher_"
DEFAULT_CLEANUP_DAYS = 7


def eprint(*args):
    print(*args, file=sys.stderr, flush=True)


def parse_xml_for_gds_id(xml_path: Path) -> str:
    tree = ET.parse(str(xml_path))
    root = tree.getroot()
    elem = root.find(".//GdsRefId")
    if elem is not None and elem.text and elem.text.strip():
        return elem.text.strip()
    raise ValueError("GDS reference ID not found in XML")


def cleanup_old_tempdirs(prefix: str = TEMP_PREFIX, days: int = DEFAULT_CLEANUP_DAYS) -> None:
    if os.environ.get("GDSLAUNCHER_NO_CLEANUP", "").strip() in ("1", "true", "TRUE", "yes", "YES"):
        eprint("[launcher] cleanup: disabled by GDSLAUNCHER_NO_CLEANUP")
        return
    base = Path(tempfile.gettempdir())
    now = time.time()
    limit = float(days) * 86400.0
    for p in base.glob(prefix + "*"):
        try:
            if p.is_dir() and (now - p.stat().st_mtime) > limit:
                shutil.rmtree(p, ignore_errors=False)
        except Exception:
            pass


class AuthClient:
    def __init__(self, base_url: str, timeout: int = 30):
        self.base_url = self._normalize_base(base_url)
        self.timeout = timeout
        self._auth_header_value: Optional[str] = None

    @staticmethod
    def _normalize_base(url: str) -> str:
        u = (url or "").strip()
        if not u:
            raise ValueError("Empty base URL")
        if not u.endswith("/"):
            u += "/"
        return u

    def _mk_url(self, path: str) -> str:
        return urljoin(self.base_url, path.lstrip("/"))

    def _request(
        self,
        method: str,
        url: str,
        body: Optional[bytes] = None,
        extra_headers: Optional[dict] = None,
        accept: Optional[str] = None,
    ):
        headers: Dict[str, str] = {}
        if accept:
            headers["Accept"] = accept
        if extra_headers:
            headers.update(extra_headers)
        if self._auth_header_value:
            headers.setdefault("Authorization", self._auth_header_value)
        req = Request(url=url, method=method.upper(), data=body, headers=headers)
        return urlopen(req, timeout=self.timeout)

    @staticmethod
    def _read_http_error(e: HTTPError) -> bytes:
        try:
            return e.read()
        except Exception:
            return b""

    def login(self, username: str, password: str) -> None:
        login_url = self._mk_url(API_LOGIN)
        body = json.dumps({"username": username, "password": password}).encode("utf-8")
        try:
            resp = self._request(
                "POST",
                login_url,
                body=body,
                extra_headers={"Content-Type": "application/json"},
                accept="application/json",
            )
            raw = resp.read()
        except HTTPError as e:
            raw = self._read_http_error(e)
            raise RuntimeError(f"Login failed: HTTP {e.code}: {raw[:500]!r}")
        except URLError as e:
            raise RuntimeError(f"Login failed: network error: {e}")

        try:
            j = json.loads(raw.decode("utf-8"))
        except Exception:
            raise RuntimeError(f"Login failed: response is not JSON: {raw[:200]!r}")

        token = None
        for k in ("token", "access_token", "jwt", "id_token"):
            v = j.get(k)
            if isinstance(v, str) and v.strip():
                token = v.strip()
                break
        if not token:
            raise RuntimeError("Login failed: token not found in response")

        self._auth_header_value = f"Bearer {token}"

    def get_json(self, path: str) -> Dict[str, Any]:
        url = self._mk_url(path)
        try:
            with self._request("GET", url, accept="application/json") as resp:
                raw = resp.read()
        except HTTPError as e:
            raw = self._read_http_error(e)
            raise RuntimeError(f"GET JSON failed: HTTP {e.code} for {url}: {raw[:500]!r}")
        except URLError as e:
            raise RuntimeError(f"GET JSON failed: network error for {url}: {e}")

        try:
            return json.loads(raw.decode("utf-8"))
        except Exception:
            raise RuntimeError(f"Invalid JSON response from {url}: {raw[:200]!r}")

    def download(self, url_or_path: str, dest_path: Path) -> None:
        url = url_or_path if url_or_path.startswith(("http://", "https://")) else self._mk_url(url_or_path)
        try:
            with self._request("GET", url) as resp:
                with dest_path.open("wb") as f:
                    while True:
                        chunk = resp.read(1024 * 256)
                        if not chunk:
                            break
                        f.write(chunk)
        except HTTPError as e:
            raw = self._read_http_error(e)
            raise RuntimeError(f"Download failed: HTTP {e.code} for {url}: {raw[:500]!r}")
        except URLError as e:
            raise RuntimeError(f"Download failed: network error for {url}: {e}")


def parse_lookup_info(lookup_json: Dict[str, Any]) -> Tuple[str, str, str, str]:
    gds_url = lookup_json.get("gdsDownloadUrl")
    gds_name = lookup_json.get("gdsFileName") or "layout.gds"
    lyp_url = lookup_json.get("lypDownloadUrl")
    lyp_name = lookup_json.get("lypFileName") or "layout.lyp"

    if not isinstance(gds_url, str) or not gds_url.strip():
        raise RuntimeError("Lookup response missing gdsDownloadUrl")
    if not isinstance(lyp_url, str) or not lyp_url.strip():
        raise RuntimeError("Lookup response missing lypDownloadUrl")

    return gds_url.strip(), str(gds_name).strip(), lyp_url.strip(), str(lyp_name).strip()


class FetchWorker(QtCore.QThread):
    progress = QtCore.Signal(str)
    success = QtCore.Signal(str)  # gds_path
    failure = QtCore.Signal(str)
    debug = QtCore.Signal(str)

    def __init__(self, gds_ref_id: str, use_mock: bool, base_url: str, username: str, password: str):
        super().__init__()
        self.gds_ref_id = gds_ref_id
        self.use_mock = use_mock
        self.base_url = base_url
        self.username = username
        self.password = password

    def run(self):
        try:
            if self.use_mock:
                self.progress.emit("Mock mode: opening local testdata")
                if self.gds_ref_id not in LOCAL_MAP:
                    raise RuntimeError(f"Unknown GdsRefId for mock mode: {self.gds_ref_id}")
                gds_path = LOCAL_MAP[self.gds_ref_id]["gds"]
                if not gds_path.exists():
                    raise RuntimeError(f"Missing bundled GDS: {gds_path}")
                self.success.emit(str(gds_path))
                return

            client = AuthClient(self.base_url, timeout=30)

            self.progress.emit("REAL: logging in ...")
            client.login(self.username, self.password)

            self.progress.emit("REAL: looking up GDS/LYP ...")
            lookup_path = API_LOOKUP.format(gdsRefId=self.gds_ref_id)
            lookup_json = client.get_json(lookup_path)

            gds_url, gds_name, lyp_url, lyp_name = parse_lookup_info(lookup_json)

            self.progress.emit("REAL: downloading GDS ...")
            tmpdir = Path(tempfile.mkdtemp(prefix=TEMP_PREFIX))
            gds_path = tmpdir / gds_name
            client.download(gds_url, gds_path)
            if not gds_path.exists() or gds_path.stat().st_size == 0:
                raise RuntimeError(f"Downloaded GDS is empty: {gds_path}")

            self.progress.emit("REAL: downloading LYP ...")
            lyp_path = tmpdir / lyp_name
            client.download(lyp_url, lyp_path)
            if not lyp_path.exists() or lyp_path.stat().st_size == 0:
                raise RuntimeError(f"Downloaded LYP is empty: {lyp_path}")

            # Also place a copy next to the GDS with the same stem (optional convenience).
            try:
                alt_lyp = gds_path.with_suffix(".lyp")
                if alt_lyp.name != lyp_path.name:
                    shutil.copy2(lyp_path, alt_lyp)
            except Exception:
                pass

            self.success.emit(str(gds_path))
            return

        except Exception:
            self.debug.emit(traceback.format_exc())
            self.failure.emit(str(sys.exc_info()[1]))


class LoginDialog(QtWidgets.QDialog):
    def __init__(self, xml_path: Path, settings: QtCore.QSettings):
        super().__init__()
        self.setWindowTitle(APP_NAME)
        self.setModal(True)

        self.settings = settings
        self.worker: Optional[FetchWorker] = None
        self.gds_path: Optional[str] = None

        try:
            self.gds_ref_id = parse_xml_for_gds_id(xml_path)
            self.init_error = None
        except Exception as e:
            self.gds_ref_id = "(invalid XML)"
            self.init_error = str(e)

        self.gds_label = QtWidgets.QLabel(f"GDS Reference ID: {self.gds_ref_id}")

        self.mock_check = QtWidgets.QCheckBox("Use mock GDS (debug)")
        self.mock_check.toggled.connect(self.on_mock_toggled)

        self.url_label = QtWidgets.QLabel("Application URL")
        self.user_label = QtWidgets.QLabel("Username")
        self.pass_label = QtWidgets.QLabel("Password")

        self.url_edit = QtWidgets.QLineEdit()
        self.url_edit.setPlaceholderText("http://127.0.0.1:8000")
        self.url_edit.setSizePolicy(QtWidgets.QSizePolicy.Expanding, QtWidgets.QSizePolicy.Fixed)

        self.copy_btn = QtWidgets.QPushButton("📋")
        self.copy_btn.setToolTip("Copy URL to clipboard")
        self.copy_btn.setFixedWidth(36)
        self.copy_btn.setFocusPolicy(QtCore.Qt.StrongFocus)
        self.copy_btn.setDefault(False)
        self.copy_btn.setAutoDefault(False)
        self.copy_btn.clicked.connect(self.on_copy_url)

        self.user_edit = QtWidgets.QLineEdit()

        self.pass_edit = QtWidgets.QLineEdit()
        self.pass_edit.setEchoMode(QtWidgets.QLineEdit.Password)

        self.login_btn = QtWidgets.QPushButton("Login & Open in KLayout")
        self.cancel_btn = QtWidgets.QPushButton("Cancel")

        self.login_btn.setDefault(False)
        self.login_btn.setAutoDefault(False)
        self.cancel_btn.setDefault(False)
        self.cancel_btn.setAutoDefault(False)

        self.status_label = QtWidgets.QLabel("")

        url_row = QtWidgets.QHBoxLayout()
        url_row.addWidget(self.url_edit, 1)
        url_row.addWidget(self.copy_btn, 0)

        form = QtWidgets.QFormLayout()
        form.setFieldGrowthPolicy(QtWidgets.QFormLayout.ExpandingFieldsGrow)
        form.addRow(self.url_label, url_row)
        form.addRow(self.user_label, self.user_edit)
        form.addRow(self.pass_label, self.pass_edit)

        btns = QtWidgets.QHBoxLayout()
        btns.addStretch()
        btns.addWidget(self.cancel_btn)
        btns.addWidget(self.login_btn)

        layout = QtWidgets.QVBoxLayout(self)
        layout.addWidget(self.gds_label)
        layout.addSpacing(6)
        layout.addWidget(self.mock_check)
        layout.addSpacing(8)
        layout.addLayout(form)
        layout.addWidget(self.status_label)
        layout.addLayout(btns)

        self.setStyleSheet("""
        QLineEdit {
            border: 1px solid #b0b0b0;
            border-radius: 4px;
            padding: 4px 6px;
        }
        QLineEdit:disabled {
            background-color: #f0f0f0;
            color: #808080;
            border: 1px solid #d0d0d0;
        }
        QLabel:disabled {
            color: #808080;
        }
        """)

        self.setMinimumWidth(720)

        self.login_btn.clicked.connect(self.on_login)
        self.cancel_btn.clicked.connect(self.reject)

        self.url_edit.installEventFilter(self)
        self.user_edit.installEventFilter(self)
        self.pass_edit.installEventFilter(self)

        self.url_edit.textChanged.connect(self.on_inputs_changed)
        self.user_edit.textChanged.connect(self.on_inputs_changed)
        self.pass_edit.textChanged.connect(self.on_inputs_changed)

        self.restore_settings()
        self.on_mock_toggled(self.mock_check.isChecked())

        if self.init_error:
            self.login_btn.setEnabled(False)
            self.status_label.setText(f"XML error: {self.init_error}")

        if self.url_edit.text().strip():
            self.user_edit.setFocus()
        else:
            self.url_edit.setFocus()

        self.on_inputs_changed()

    def eventFilter(self, obj, event):
        if event.type() == QtCore.QEvent.KeyPress and event.key() in (QtCore.Qt.Key_Return, QtCore.Qt.Key_Enter):
            if self.mock_check.isChecked():
                event.accept()
                return True

            if obj == self.url_edit:
                QtCore.QTimer.singleShot(0, self.user_edit.setFocus)
                event.accept()
                return True

            if obj == self.user_edit:
                QtCore.QTimer.singleShot(0, self.pass_edit.setFocus)
                event.accept()
                return True

            if obj == self.pass_edit:
                if self._inputs_complete_real():
                    QtCore.QTimer.singleShot(0, self.login_btn.setFocus)
                else:
                    self.status_label.setText("Fill Application URL, Username, Password.")
                    QtCore.QTimer.singleShot(0, self.pass_edit.setFocus)
                event.accept()
                return True

        return super().eventFilter(obj, event)

    def keyPressEvent(self, event):
        if event.key() in (QtCore.Qt.Key_Return, QtCore.Qt.Key_Enter):
            if self.focusWidget() == self.login_btn:
                self.on_login()
                event.accept()
                return
            event.accept()
            return
        super().keyPressEvent(event)

    def on_copy_url(self):
        text = self.url_edit.text()
        if not text:
            return
        QtWidgets.QApplication.clipboard().setText(text)
        self.status_label.setText("URL copied to clipboard.")

    def restore_settings(self) -> None:
        self.settings.beginGroup("ui")
        try:
            url = self.settings.value("url", "", type=str)
            user = self.settings.value("username", "", type=str)
            use_mock = self.settings.value("use_mock", False, type=bool)
            self.url_edit.setText(url)
            self.user_edit.setText(user)
            self.mock_check.setChecked(use_mock)

            geom = self.settings.value("geometry")
            if geom is not None:
                self.restoreGeometry(geom)
        finally:
            self.settings.endGroup()

    def save_settings(self) -> None:
        self.settings.beginGroup("ui")
        try:
            self.settings.setValue("url", self.url_edit.text().strip())
            self.settings.setValue("username", self.user_edit.text().strip())
            self.settings.setValue("use_mock", self.mock_check.isChecked())
            self.settings.setValue("geometry", self.saveGeometry())
        finally:
            self.settings.endGroup()
        self.settings.sync()

    def _inputs_complete_real(self) -> bool:
        url = self.url_edit.text().strip()
        user = self.user_edit.text().strip()
        pw = self.pass_edit.text()
        if not url or not user or not pw:
            return False
        if not (url.startswith("http://") or url.startswith("https://")):
            return False
        return True

    def on_inputs_changed(self):
        if self.init_error:
            return
        if self.mock_check.isChecked():
            self.login_btn.setEnabled(True)
            return
        self.login_btn.setEnabled(self._inputs_complete_real())
        self.login_btn.setDefault(False)
        self.login_btn.setAutoDefault(False)

    def on_mock_toggled(self, checked: bool):
        for w in (self.url_edit, self.user_edit, self.pass_edit, self.copy_btn):
            w.setEnabled(not checked)
        for l in (self.url_label, self.user_label, self.pass_label):
            l.setEnabled(not checked)

        if self.init_error:
            return

        if checked:
            self.login_btn.setText("Open in KLayout")
            self.status_label.setText("Mock mode: no login required.")
            self.pass_edit.setText("")
            self.login_btn.setEnabled(True)
        else:
            self.login_btn.setText("Login & Open in KLayout")
            self.status_label.setText("")
            self.on_inputs_changed()

    def set_busy(self, busy: bool):
        for w in (
            self.mock_check,
            self.url_edit,
            self.user_edit,
            self.pass_edit,
            self.copy_btn,
            self.login_btn,
            self.cancel_btn,
        ):
            w.setEnabled(not busy)

        if not busy:
            self.on_mock_toggled(self.mock_check.isChecked())

        if self.init_error:
            self.login_btn.setEnabled(False)

    def on_login(self):
        if self.init_error:
            return

        use_mock = self.mock_check.isChecked()
        self.save_settings()

        if not use_mock and not self._inputs_complete_real():
            QtWidgets.QMessageBox.warning(self, APP_NAME, "Please fill: Application URL, Username, Password")
            return

        self.set_busy(True)
        self.status_label.setText("Starting...")

        url = self.url_edit.text().strip()
        user = self.user_edit.text().strip()
        pw = self.pass_edit.text()

        self.worker = FetchWorker(self.gds_ref_id, use_mock, url, user, pw)
        self.worker.progress.connect(self.status_label.setText)
        self.worker.success.connect(self.on_success)
        self.worker.failure.connect(self.on_failure)
        self.worker.debug.connect(lambda s: eprint(s))
        self.worker.start()

    def on_success(self, gds_path: str):
        self.gds_path = gds_path
        self.set_busy(False)
        self.accept()

    def on_failure(self, msg: str):
        self.set_busy(False)
        QtWidgets.QMessageBox.critical(self, APP_NAME, msg)

    def closeEvent(self, event):
        try:
            self.save_settings()
        except Exception:
            pass
        super().closeEvent(event)


def launch_klayout(gds_path: str):
    cmd = os.environ.get("KLAYOUT_CMD", DEFAULT_KLAYOUT_CMD)

    gds_p = Path(gds_path)
    lyp_p = gds_p.with_suffix(".lyp")

    if lyp_p.exists():
        eprint(f"[launcher] launching KLayout: {cmd} -l {str(lyp_p)!r} {str(gds_p)!r}")
        subprocess.Popen([cmd, "-l", str(lyp_p), str(gds_p)])
    else:
        eprint(f"[launcher] launching KLayout: {cmd} {str(gds_p)!r}")
        subprocess.Popen([cmd, str(gds_p)])


def main(argv):
    eprint("[launcher] started")
    cleanup_old_tempdirs()

    if len(argv) < 2:
        print("Usage: Launcher.py <input.xml>", file=sys.stderr)
        return 2

    xml_path = Path(argv[1]).expanduser().resolve()
    if not xml_path.exists():
        print(f"XML not found: {xml_path}", file=sys.stderr)
        return 2

    QtCore.QCoreApplication.setOrganizationName(ORG_NAME)
    QtCore.QCoreApplication.setApplicationName(APP_SETTINGS_NAME)
    settings = QtCore.QSettings()

    app = QtWidgets.QApplication(argv)
    app.setApplicationName(APP_NAME)

    dlg = LoginDialog(xml_path, settings)
    rc = dlg.exec()
    if rc != QtWidgets.QDialog.Accepted:
        return 0

    if dlg.gds_path:
        launch_klayout(dlg.gds_path)
        QtWidgets.QMessageBox.information(None, APP_NAME, "KLayout launched.")
    return 0


if __name__ == "__main__":
    sys.exit(main(sys.argv))
